
/************************************************************************
 *
 * \file: AoapTransport.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android Auto - Prototype
 *
 * \author: D. Girnus / ADIT/SW1/Brunel / dgirnus@de.adit-jv.com
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/


/* *************  includes  ************* */

#include <adit_logging.h>
#include <pthread_adit.h>
#include <adit_typedef.h>
#include <sys_time_adit.h>
#include <stdint.h>
#include <stdio.h>

#include <sys/unistd.h>
#include <sys/types.h>

#include <fcntl.h>

#include <assert.h>

#include "AoapTransport.h"

/* AOAP library header files */
#include <aoap_types.h>
#include <aoap.h>

/* *************  defines  ************* */

LOG_IMPORT_CONTEXT(aauto_transport)

/* buffer size of the local read buffer */
#define MAX_READ                        4096

using namespace std;

/* *************  functions  ************* */

namespace adit { namespace aauto {


int AoapDevice::switchDevice(aoapDeviceInfo_t* inAoapDeviceInfo, aoapTransportInfo_t* outTransportInfo, unsigned int inTimeout) {
    int ret = 0;
    
    if (outTransportInfo == nullptr || inAoapDeviceInfo == nullptr) {
        LOG_ERROR((aauto_transport, "%s() Invalid parameter outTransportInfo=%p, inAoapDeviceInfo=%p",
                __FUNCTION__, outTransportInfo, inAoapDeviceInfo));
        return -1;
    }

    /* create SyncContext. will be deleted by itself */

    /* create AOAP library accessory session */
    outTransportInfo->aoapAccessoryId = aoap_create_accessory((t_aoap_accessory_param*)&inAoapDeviceInfo->aoapAccessoryInfo);
    if (outTransportInfo->aoapAccessoryId < 0) {
        LOG_ERROR((aauto_transport, "%s() Create libaoap accessory session failed=%d",
                    __FUNCTION__, outTransportInfo->aoapAccessoryId));
        ret = -1;
    } else {
        if (0 == inTimeout) {
            LOG_WARN((aauto_transport, "%s() timeout is=%d. Use timeout > 0.", __FUNCTION__, inTimeout));
            inTimeout = 8;
            LOG_WARN((aauto_transport, "%s() Adjust timeout to %d", __FUNCTION__, inTimeout));
        }
        /* change connect timeout based on the Application setting */
        aoap_set_connect_timeout(outTransportInfo->aoapAccessoryId, inTimeout / 1000);

        /* switch to accessory mode */
        outTransportInfo->aoapDeviceId = aoap_connect(outTransportInfo->aoapAccessoryId, \
                                                      inAoapDeviceInfo->vendorId, inAoapDeviceInfo->productId, inAoapDeviceInfo->pSerial, \
                                                      &switchCallback, inAoapDeviceInfo->aoapAccessoryInfo.enableAudio, this);
        if (0 > outTransportInfo->aoapDeviceId) {
            LOG_ERROR((aauto_transport, "%s() Create libaoap connection failed=%d", __FUNCTION__, outTransportInfo->aoapDeviceId));
            ret = -1;
        } else {
            /* switch to accessory mode initiated.
             * wait for AOAP library switch callback (condition to become true) */
            if (0 > sem_wait(&mAoapSyncSem)) {
                LOG_ERROR((aauto_transport, "%s() sem_wait() failed", __FUNCTION__));
            }
            if (0 == mSwitchResult) {
                LOG_INFO((aauto_transport, "%s() Device with serial=%s switched to accessory mode =%d",
                        __FUNCTION__, inAoapDeviceInfo->pSerial, mSwitchResult));
                /* return switch result */
                ret = mSwitchResult;
            } else if (AOAP_ERROR_ALREADY_DONE == mSwitchResult) {
                LOG_INFO((aauto_transport, "%s() Device with serial=%s re-connected to accessory, aoapDeviceId =%d",
                        __FUNCTION__, inAoapDeviceInfo->pSerial, outTransportInfo->aoapDeviceId));
                ret = 0;
            } else {
                LOG_ERROR((aauto_transport, "%s() Switch device with serial=%s to accessory mode failed =%d",
                        __FUNCTION__, inAoapDeviceInfo->pSerial, mSwitchResult));
                ret = mSwitchResult;
            }
        }
        /* de-initialize AOAP library in case switch failed */
        if (0 > ret) {
            LOG_INFO((aauto_transport, "%s() Cleanup AOAP library accessory session due to switch device failed.", __FUNCTION__));

            /* cleanup AOAP library accessory session
             * if no devices associated to the accessory session. */
            aoap_defer_delete_accessory(outTransportInfo->aoapAccessoryId, outTransportInfo->aoapDeviceId);
            /* did not set aoapAccessoryId to -1 to support multiple AAP devices */
            outTransportInfo->aoapDeviceId = -1;
        }
    }

    return ret;
}

void AoapDevice::switchCallback(int accessoryId, int deviceId, int result, void *pToken, unsigned int audioSupport) {
    (void)audioSupport;

    if (AOAP_ERROR_ALREADY_DONE == result) {
        LOG_INFO((aauto_transport, "%s() Switch to accessory mode (aoapAccessoryId =%d, aoapDeviceId =%d) done =%d",
                __FUNCTION__, accessoryId, deviceId, result));
        /* Note: If result is AOAP_ERROR_ALREADY_DONE
         * Expect that callback was triggered by API aoap_connect()->connectDevice()
         * and aoap_connect() return in switchDevice() with valid value.
         * In this case, do not use SyncContext for synchronization! */
    } else if (0 > result) {
        LOG_ERROR((aauto_transport, "%s() Switch to accessory mode (aoapAccessoryId =%d, aoapDeviceId =%d) failed =%d",
                __FUNCTION__, accessoryId, deviceId, result));
    } else {
        LOG_INFO((aauto_transport, "%s() Switch to accessory mode (aoapAccessoryId =%d, aoapDeviceId =%d) success =%d",
                __FUNCTION__, accessoryId, deviceId, result));
    }
    if (NULL != pToken) {
        AoapDevice* me = (AoapDevice*)pToken;
        me->mSwitchResult = result;
        if (0 > sem_post(&me->mAoapSyncSem)) {
            LOG_ERROR((aauto_transport, "%s() sem_post() failed (aoapAccessoryId =%d, aoapDeviceId =%d)",
                    __FUNCTION__, accessoryId, deviceId));
        }
    } else {
        LOG_ERROR((aauto_transport, "%s() pToken is NULL. Cannot signal condition (aoapAccessoryId =%d, aoapDeviceId =%d)",
                __FUNCTION__, accessoryId, deviceId));
        assert(pToken);
    }
}

AoapTransport::AoapTransport(shared_ptr<GalReceiver> inReceiver,
        aoapTransportInfo_t* inAoapTransportInfo) :
        Transport(inReceiver) {
    /* set the read buffer parameter to default values */
    mReadBuffer.buf = nullptr;
    mReadBuffer.size = 0;
    mReadBuffer.pos = 0;
    mReadBuffer.bytesRemaining = 0;

    if(inAoapTransportInfo != nullptr) {
        /* set the AOAP library parameter to the values
         * received by AoapDevice::switchDevice */
        mAoapAccId = inAoapTransportInfo->aoapAccessoryId;
        mAoapDevId = inAoapTransportInfo->aoapDeviceId;
    } else {
        mAoapAccId = -1;
        mAoapDevId = -1;
        LOG_ERROR((aauto_transport, "%s() Invalid parameter inAoapTransportInfo=%p",
                __FUNCTION__, inAoapTransportInfo));
    }
}

AoapTransport::~AoapTransport() {
    if (nullptr != mReadBuffer.buf) {
        delete[] mReadBuffer.buf;
        mReadBuffer.buf = nullptr;
    }
    mReadBuffer.size = 0;
    mReadBuffer.pos = 0;
    mReadBuffer.bytesRemaining = 0;

    if (mAoapAccId > 0) {
        LOG_INFO((aauto_transport, "%s() cleanup AOAP library accessory session", __FUNCTION__));
        /* cleanup AOAP library accessory session
         * if no devices associated to the accessory session. */
        aoap_defer_delete_accessory(mAoapAccId, mAoapDevId);
        mAoapAccId = -1;
        mAoapDevId = -1;
    }

}

void AoapTransport::abortReads() {
    /* Nothing to do */
}

int AoapTransport::read(void* buf, size_t len) {

    int ret = 0;
    size_t bytesCopied = 0;

    if ((0 >= mAoapAccId) && (0 >= mAoapDevId) ) {
        LOG_ERROR((aauto_transport, "%s() mAoapAccId=%d, mAoapDevId=%d are invalid",
                    __FUNCTION__, mAoapAccId, mAoapDevId));
        return TRANSPORT_ERROR_FATAL;
    }

    /* Allocate the internal buffer. */
    if (nullptr == mReadBuffer.buf) {
        mReadBuffer.size = MAX_READ;
        mReadBuffer.buf = new unsigned char[mReadBuffer.size];
        if (mReadBuffer.buf) {
            mReadBuffer.bytesRemaining = 0;
            mReadBuffer.pos = 0;
            memset(mReadBuffer.buf, 0, mReadBuffer.size);
        } else {
            LOG_ERROR((aauto_transport, "%s() cannot allocate mReadBuffer", __FUNCTION__));
            return TRANSPORT_ERROR_FATAL;
        }
    }

    /* Reached end of buffer. */
    if (mReadBuffer.pos >= MAX_READ) {
        mReadBuffer.pos = 0;
    }

    /* Check if it necessary to read from the AOAP transport. */
    if ((0 == mReadBuffer.pos) || (0 >= mReadBuffer.bytesRemaining)) {
        unsigned int transferred = 0;
        int err = aoap_read1(mAoapAccId, mAoapDevId, &mReadBuffer.buf[0], mReadBuffer.size, &transferred, 1000);

        LOGD_VERBOSE((aauto_transport, "%s() = %d (msg len: %zu, transferred: %u)", __FUNCTION__, err, len, transferred));

        /* Validate if read was successfull and notify upper layer about status */
        ret = validateTransfer(err, false);
        if ( (TRANSPORT_ERROR_FATAL != ret) && (transferred > 0) ) {
            mReadBuffer.pos = 0;
            mReadBuffer.bytesRemaining = transferred;
        }
    }

    /* Use the internal buffer to provide chunks of AOAP transport data. */
    if (mReadBuffer.bytesRemaining >= len) {
        LOGD_VERBOSE((aauto_transport, "%s() read from mReadBuffer(mPos=%zu, len=%zu)",
                    __FUNCTION__, mReadBuffer.pos, len));
        bytesCopied = len;
        memcpy(buf, &mReadBuffer.buf[mReadBuffer.pos], bytesCopied);
        mReadBuffer.pos += bytesCopied;
        mReadBuffer.bytesRemaining -= bytesCopied;

        ret = bytesCopied;
    } else if (mReadBuffer.bytesRemaining > 0) {
        LOGD_VERBOSE((aauto_transport, "%s() read last available bytes from mReadBuffer(mPos=%zu, mBytesRemaining=%zu, len=%zu)",
                __FUNCTION__, mReadBuffer.pos, mReadBuffer.bytesRemaining, len));
        bytesCopied = mReadBuffer.bytesRemaining;
        memcpy(buf, &mReadBuffer.buf[mReadBuffer.pos], bytesCopied);
        mReadBuffer.pos += bytesCopied;
        mReadBuffer.bytesRemaining -= bytesCopied;

        ret = bytesCopied;
    } else {
        /* Return value was set by validateTransfer() */
    }

    return ret;
}

int AoapTransport::write(void* buf, size_t len) {
    if ((0 >= mAoapAccId) && (0 >= mAoapDevId) ) {
        LOG_ERROR((aauto_transport, "%s() mAoapAccId=%d, mAoapDevId=%d are invalid",
                __FUNCTION__, mAoapAccId, mAoapDevId));
        return TRANSPORT_ERROR_FATAL;
    }

    unsigned int transferred = 0;
    int err = aoap_write1(mAoapAccId, mAoapDevId, (unsigned char*)buf, len, &transferred, 1000);

    LOGD_VERBOSE((aauto_transport, "%s() = %d (msg len: %zu, transferred: %u)", __FUNCTION__, err, len, transferred));

    /* Validate if write was successfull and notify upper layer about status */
    int ret = validateTransfer(err, true);
    if ( (TRANSPORT_ERROR_FATAL != ret) && (transferred > 0) ) {
        ret = transferred;
    }

    return ret;
}

int32_t AoapTransport::validateTransfer(int32_t inResult, bool inWasWriteRequest)
{
    GalReceiverTransportStatus notify = GalReceiverTransportStatus::UNKNOWN_ERR;
    int32_t ret = 0;

    switch (inResult)
    {
        case AOAP_SUCCESS:
        {
            notify = (inWasWriteRequest==true)?GalReceiverTransportStatus::WRITE_SUCCESS:GalReceiverTransportStatus::READ_SUCCESS;
            break;
        }
        case AOAP_ERROR_IO:
        {
            if (inWasWriteRequest == true) {
                LOGD_VERBOSE((aauto_transport, "I/O error (%d) occurred during write", inResult));
                notify = GalReceiverTransportStatus::WRITE_ERR_IO;
            } else {
                LOGD_VERBOSE((aauto_transport, "I/O error (%d) occurred during read", inResult));
                notify = GalReceiverTransportStatus::READ_ERR_IO;
            }
            ret = TRANSPORT_ERROR_NONFATAL;
            break;
        }
        case AOAP_ERROR_TIMEOUT:
        {
            if (inWasWriteRequest == true) {
                LOG_WARN((aauto_transport, "Timeout (%d) occurred during write", inResult));
                notify = GalReceiverTransportStatus::WRITE_ERR_TIMEOUT;
                ret = TRANSPORT_ERROR_NONFATAL;
            } else {
                /* Do not notify about read timeouts. */
                /* Will occur almost once per second, if there is no data comming in
                   from the device (normal behavior) */
                LOGD_DEBUG((aauto_transport, "Timeout (%d) occurred during read", inResult));
                notify = GalReceiverTransportStatus::READ_SUCCESS;
                ret = 0;
            }
            break;
        }
        default:
        {
            if (inWasWriteRequest == true) {
                LOG_ERROR((aauto_transport, "Error (%d) occurred during write", inResult));
                notify = GalReceiverTransportStatus::WRITE_ERR_FATAL;
            } else {
                LOG_ERROR((aauto_transport, "Error (%d) occurred during read", inResult));
                notify = GalReceiverTransportStatus::READ_ERR_FATAL;
            }
            ret = TRANSPORT_ERROR_FATAL;
            break;
        }
    }
    /* Notify about current status if status has been changed. */
    notifyStatus(notify);

    return ret;
}

} } /* namespace adit { namespace aauto { */
